/*

characterstudio.cpp

Character studio related export stuff should be migrated here

*/

#include "stdafx.h"

#ifdef CSSUPPORT
#include "phyexp.h"
#include "bipexp.h"
#endif

#include "DffExp.h"
#include "characterstudio.h"
#include "utilities.h"
#include "matrixmangle.h"
#include "keysample.h"

#include "physique2and3.h"

#include "osutils.h"

static PhysiqueHandle *s_physiqueHandle = 0;

PhysiqueHandle *CreatePhysiqueHandle( Interface *gi, WarningList *warningList )
//we must determine physique version in a very convoluted way
//- see physique2and3.cpp for full explanation
{
    //scan through all dlls max loaded
    DllDir & dir = gi->GetDllDir();    
    for (int i=0; i<dir.Count(); i++) 
    {
        //get a description of the ith dll
        DllDesc & dllDesc = dir[i];

        if (dllDesc.loaded)
        {
            std::string desc =  dllDesc.Description();

            //if (desc == std::string("Physique Modifier: a skeletal deformation tool"))
            if (stricmp(dllDesc.fname,"physique.dlm")==0) 
            {
                //it's the physique modifier dll - dig out the version info from windows!
                std::string version = GetFileVersion( dllDesc.fname, MAKELANGID(LANG_ENGLISH,SUBLANG_ENGLISH_US) );

                //record the version number to log file
                std::string physiqueversion =
                    std::string("Using physique version ") + version;

                warningList->add(
                    CWarning(wtInformational,"",physiqueversion.c_str())
                    );

                //instance a physique handle for correct version of physique
                int majorVersion = version.c_str()[0] - '0';

                if (majorVersion >= 3)
                {
                    return new Physique3Handle();
                }
                else
                {
                    return new Physique2Handle();
                }
            }
        }
    }

    return 0;
}

bool IsABipedNode( INode *node )
{
#ifdef CSSUPPORT
    Control *TMController = node->GetTMController();
    if (TMController)
    {
        if ((TMController->ClassID() == BIPSLAVE_CONTROL_CLASS_ID) ||
            (TMController->ClassID() == BIPBODY_CONTROL_CLASS_ID) ||
            (TMController->ClassID() == FOOTPRINT_CLASS_ID))
        {
            return true;
        }
    }
#endif
    return false;
}

static
bool ParentIsntABipedNode( INode *node )
{
    return !IsABipedNode( node->GetParentNode() );
}

bool
IsAFootstepNode( INode *node )
{
#ifdef CSSUPPORT
    Control *TMController = node->GetTMController();
    if (TMController)
    {
        if (TMController->ClassID() == FOOTPRINT_CLASS_ID)
        {
            return true;
        }
    }
#endif
    return false;
}


/* Standardise tagging for biped's should allow animations to
   be shared between different biped's, possibly even ones with
   different structure */
#define BIPEDTAGOFFSET 1000
#define MAXBIPEDTAG 105
static char *BipedTagNames[] =
{
    "ROOT",
    "Head",
    "L Calf",
    "L Clavicle",
    "L Finger0",
    "L Finger1",
    "L Finger01",
    "L Finger02",
    "L Finger2",
    "L Finger3",
    "L Finger4",
    "L Finger11",
    "L Finger12",
    "L Finger21",
    "L Finger22",
    "L Finger31",
    "L Finger32",
    "L Finger41",
    "L Finger42",
    "L Foot",
    "L Forearm",
    "L Hand",
    "L HorseLink",
    "L Thigh",
    "L Toe0",
    "L Toe1",
    "L Toe01",
    "L Toe2",
    "L Toe02",
    "L Toe3",
    "L Toe4",
    "L Toe11",
    "L Toe12",
    "L Toe21",
    "L Toe22",
    "L Toe31",
    "L Toe32",
    "L Toe41",
    "L Toe42",
    "L UpperArm",
    "Neck",
    "Neck1",
    "Neck2",
    "Neck3",
    "Neck4",
    "Pelvis",
    "Ponytail1",
    "Ponytail2",
    "Ponytail11",
    "Ponytail12",
    "Ponytail13",
    "Ponytail14",
    "Ponytail21",
    "Ponytail22",
    "Ponytail23",
    "Ponytail24",
    "R Calf",
    "R Clavicle",
    "R Finger0",
    "R Finger1",
    "R Finger01",
    "R Finger02",
    "R Finger2",
    "R Finger3",
    "R Finger4",
    "R Finger11",
    "R Finger12",
    "R Finger21",
    "R Finger22",
    "R Finger31",
    "R Finger32",
    "R Finger41",
    "R Finger42",
    "R Foot",
    "R Forearm",
    "R Hand",
    "R HorseLink",
    "R Thigh",
    "R Toe0",
    "R Toe1",
    "R Toe01",
    "R Toe2",
    "R Toe02",
    "R Toe3",
    "R Toe4",
    "R Toe11",
    "R Toe12",
    "R Toe21",
    "R Toe22",
    "R Toe31",
    "R Toe32",
    "R Toe41",
    "R Toe42",
    "R UpperArm",
    "Spine",
    "Spine1",
    "Spine2",
    "Spine3",
    "Spine4",
    "Tail",
    "Tail1",
    "Tail2",
    "Tail3",
    "Tail4",
    "Footsteps"
};

int
DFFExport::GetNodeTagValue(INode *node)
{
    int tag = -1;

    int TagID;

    if (node->GetUserPropInt("TAG", TagID) ||
        node->GetUserPropInt("tag", TagID))
    {
        return TagID;
    }

    if (m_CSExport)
    {
        /* try and find it in the standard names */
        char *name = node->GetName();
        RwBool standardName = FALSE;
        if (CSBipedName && strncmp(CSBipedName, name, strlen(CSBipedName)) == 0)
        {
            if (strcmp(CSBipedName, name) == 0)
            {
                /* Root of biped  */
                standardName = TRUE;
                tag = BIPEDTAGOFFSET;
            }
            else
            {
                /* Check all other standard part names */
                int i;
                for (i=1; i<MAXBIPEDTAG; i++)
                {
                    if (strcmp(&name[strlen(CSBipedName) + 1], BipedTagNames[i]) == 0)
                    {
                        standardName = TRUE;
                        tag = i+BIPEDTAGOFFSET;
                    }
                }
            }
        }
        if (!standardName)  //we make it a bone anyway?
        {
            tag = CSExtraBoneTagIndex;
            CSExtraBoneTagIndex++;
        }
    }
    else
    {
        tag = CSExtraBoneTagIndex;
        CSExtraBoneTagIndex++;            
    }

    return tag;
}

char *
FindBipedNameInHierarchy(INode *root)
//Depth-first search from root.
//Return a strdup of the name of the first node we find with a biped controller.
//(mind the leaks!)
{
#ifdef CSSUPPORT
    Control *c = root->GetTMController();
    char *name = root->GetName();
    if (c &&
        c->ClassID() == BIPBODY_CONTROL_CLASS_ID)
    {
        return (strdup(root->GetName()));
    }
    else
    {
        int numChildren = root->NumberOfChildren();
        int i;

        for (i=0; i<numChildren; i++)
        {
            char *name = NULL;

            name = FindBipedNameInHierarchy(root->GetChildNode(i));
            if (name)
            {
                return name;
            }
        }

        return NULL;
    }
#else
    return NULL;
#endif
}

void
DFFExport::AddTopLevelBipedPartKeys(INode *nodeHierarchyRoot, ANMInfo *ANMData, RwInt32 m_tvStart,
                                    RwInt32 m_tvEnd, RwInt32 m_nNumFrames)
{
    char *TopLevelPartNames[] =
    {
        "L Clavicle",         /* 0 */
        "R Clavicle",
        "L Finger0",
        "R Finger0",
        "L Thigh",
        "R Thigh",
        "L Toe0",
        "R Toe0",
        "Spine",
        "Tail",
        "Head",
        "Pelvis",
        "Footsteps",
        "Neck",
        "Ponytail1",
        "Ponytail2"
    };

    char nodename[1024];
    INode *node;
    int part;

    for (part=0; part<16; part++)
    {
        sprintf(nodename, "%s %s", CSBipedName, TopLevelPartNames[part]);
        node = findNodeByName(nodeHierarchyRoot, nodename);
        if (node)
        {
            Control *control = node->GetTMController();

            if (control)
            {
                FindControlKeys(control, ANMData->keys, m_tvStart, m_tvEnd, m_noExtraInterpKeys);
            }
        }
        else
        {
            RWEXPMESSAGE(("%s(%d): Couldn't find biped part : %s %s\n",
                          __FILE__, __LINE__,
                          CSBipedName, TopLevelPartNames[part]));
        }
    }
}

Modifier * 
FindPhysiqueModifier (INode* nodePtr)
//search this node's modifier stack for a physique modifier & return a pointer to it,
//if none exists, return NULL.
{
//    char name[256];
//    strcpy(name, nodePtr->GetName() );

#ifdef CSSUPPORT
    return FindModifier( nodePtr, Class_ID(PHYSIQUE_CLASS_ID_A, PHYSIQUE_CLASS_ID_B) );
#else
    return NULL;
#endif
}

bool HasAPhysiqueModifier( INode *node )
{
    return FindPhysiqueModifier( node ) != 0;
}

static INode *FindBiped( INode *physiqueNode )
{
    //could cache this...
    BonesContainer *bones = s_physiqueHandle->GetInfluentialBonesFromPhysique( physiqueNode );

    //really shouldn't happen!
    if ((bones == 0) || bones->empty())
    {
        delete bones;

        return physiqueNode;
    }

    //any bone will do for now.
    INode *bone = *(bones->begin());

    delete bones;

    return bone;
}

void
DFFExport::ProcessPhysiqueNode(INode *node, INode *rootNode, RpClump *clump)
{
    Modifier *physiqueMod = FindPhysiqueModifier(node);
    if (!physiqueMod)
    {
        //not a physique, forget it!
        return;
    }

    //we need to transform the skin into the correct space relative to the bones
    //& renderware friendly so work out some matrices to do it.
    Matrix3 skinMat = node->GetObjTMAfterWSM(m_tvStart);

    //Get a bone node connected to this physique, then find the highest biped part
    //in that bone's hierarchy.
    //Won't work with floating bones of course, but otherwise should be okay.
    INode *boneNode = FindFirstAncestorIf( FindBiped(node), ParentIsntABipedNode );
    
    Matrix3 transform;
    
    //don't ask why, it just works. :-P
    if (boneNode->GetParentNode()->IsRootNode())
    {
        //bone is top of hierarchy

        //get the bone's matrix and knock out any weird scalings
        Matrix3 boneMat = boneNode->GetNodeTM(m_tvStart);
        boneMat.NoScale();

/*        transform = skinMat * Inverse(boneMat)
            //(magic matrix wangles Biped space into Renderware space)
            * Matrix3( Point3(0,0,1),
                       Point3(1,0,0),
                       Point3(0,1,0),
                       Point3(0,0,0) );*/
        if (m_retainObjInWorldSpace)
        {
            Matrix3 parentMat = node->GetParentNode()->GetObjTMAfterWSM(m_tvStart);
            parentMat.NoScale();

            transform = skinMat * Inverse( parentMat );
        }
        else
        {
            transform = skinMat *  Inverse(boneMat);
        }

    }
    else
    {
        //bone is a child of something else
        Matrix3 mat = FindTopOfHierarchy( boneNode )->GetObjTMAfterWSM(m_tvStart);
        transform = skinMat * Inverse(mat);
    }
    
    //don't forget to adjust the skin for the user scale!
    transform *= m_scaleMatrix;

    //Get the vertices & normals & transform them into the correct space...
    RwInt32 *vertexMap = NULL;
    RpAtomic *atomic = NULL;

    ExtractGeometryFromNode( node, clump, RpClumpGetFrame(clump), atomic, vertexMap, transform );

    //Get the vertex->bone mappings and build an RpSkin atomic
    if (vertexMap)
    {
        if (atomic)
        {
           RwMatrixWeights  *skinMatrixWeights = 0;
           RwUInt32         *skinVertexIndices = 0;

           s_physiqueHandle->CalculateCSVertexBoneMap(node, vertexMap, CSNumBones, CSBoneIndexNodePointers,
               m_remapper,
               skinMatrixWeights, skinVertexIndices );

           RpSkin *skin = RpSkinCreate(atomic, CSNumBones, skinMatrixWeights, skinVertexIndices, skinInverseMatrices, skinFlags);                
           if (skin)
           {
               int i;
               /* set the bonetags to match the frame hierarchy */
               for (i=0; i<CSNumBones; i++)
               {                    
                   skin->pBoneInfo[i].boneTag = CSBoneIndexTags[i];
               }
           }
       
           RwFree(skinMatrixWeights);
           RwFree(skinVertexIndices);
        }

        RwFree(vertexMap);
    }
}

void ProcessPhysiqueNodes( DFFExport *dffExp, SkinsContainer & skins, INode *rootNode, RpClump *clump )
{
    for (SkinsContainer::iterator i = skins.begin(); i != skins.end(); ++i)
    {
        dffExp->ProcessPhysiqueNode( *i, rootNode, clump );
    }
}

struct AddToSkinsIfPhysique
{
    AddToSkinsIfPhysique( SkinsContainer & skins ) : m_skins( skins ) {}

    void operator()( INode *node )
    { 
        if (HasAPhysiqueModifier(node))
        {
            m_skins.insert( node );
        }
    }

    SkinsContainer & m_skins;
};

struct AddToBonesIfBipedBone
{
    AddToBonesIfBipedBone( BonesContainer & bones ) : m_bones( bones ) {}

    void operator()( INode *node )
    {
        if (IsABipedNode(node) && !IsAFootstepNode(node))
        {
            m_bones.insert( node );
        }
    }

    BonesContainer & m_bones;
};

#include <map>
#include <algorithm>

typedef std::map<INode *, BonesContainer *> PhysiquesToBonesMap;

struct MapPhysiquesToInfluenceBones
{
    MapPhysiquesToInfluenceBones( PhysiquesToBonesMap &map ) : m_map( map ) {}

    void operator()( INode *node )
    {
        if (HasAPhysiqueModifier(node))
        {
            BonesContainer *bones = s_physiqueHandle->GetInfluentialBonesFromPhysique( node );

            if (bones != 0)
            {
                m_map[ node ] = bones;
            }
        }
    }

    PhysiquesToBonesMap & m_map;
};

struct InsertIntoBonesContainer
{
    InsertIntoBonesContainer( BonesContainer & bones ) : m_bones(bones) {}

    void operator ()(INode *node)
    {
        m_bones.insert( node );
    }

    BonesContainer & m_bones;
};

struct AddInfluenceBonesHeirachiesToHeirarchyBones
{
    AddInfluenceBonesHeirachiesToHeirarchyBones( PhysiquesToBonesMap &map, BonesContainer & bones )
        : m_map( map ), m_bones(bones) {}

    //Make sure all selected physique influencing bones' hierarchies are in there to heirarchy bones
    //(so we'll find all the other physiques that are attached to them!)
    void operator()( INode *node )
    {
        //node will be in map iff it has a physique modifier
        PhysiquesToBonesMap::iterator p = m_map.find(node);

        if (p == m_map.end())
        {
            return; //not there
        }

        //it's there, get it's influence bones.
        BonesContainer *influenceBones = p->second;

        for (BonesContainer::iterator i = influenceBones->begin();
            i != influenceBones->end(); ++i)
        {
            //make sure these bone's heirachies are all in our hierarchy bones
            BonesContainer::iterator h = m_bones.find( *i );

            if (h == m_bones.end())
            {
                //not there, add it's heirarchy if that's not there either:
                INode *hierarchy = FindTopOfHierarchy( *i );

                h = m_bones.find( hierarchy );

                if (h == m_bones.end())
                {
                    ForEachNodeInHierarchy( hierarchy,
                        InsertIntoBonesContainer( m_bones ) );
                }
            }
        }        
    }

    PhysiquesToBonesMap &m_map;
    BonesContainer & m_bones;
};

struct AddToSkinsIfInfluencedByBone
{
    AddToSkinsIfInfluencedByBone(
        SkinsContainer & skins,
        BonesContainer & bones,
        PhysiquesToBonesMap & map )
        : m_skins(skins), m_hierarchyBones(bones), m_map(map) {}

    void operator()( INode *node )
    //Add this node if it's a physique that is influenced by some hierarchy bone
    //(and it's not in there already)
    {
        //node will be in map iff it has a physique modifier
        PhysiquesToBonesMap::iterator p = m_map.find(node);

        if (p == m_map.end())
        {
            return; //not there
        }

        //it's there, get it's influence bones.
        BonesContainer *influenceBones = p->second;

        //if any influence bone is a heirarchy bone, export this skin!
        for (BonesContainer::iterator i = influenceBones->begin();
            i != influenceBones->end(); ++i)
        {
            if (m_hierarchyBones.find( *i ) != m_hierarchyBones.end())
            {
                m_skins.insert( node );
                return;
            }
        }
    }

    SkinsContainer & m_skins;
    BonesContainer & m_hierarchyBones;
    PhysiquesToBonesMap & m_map;
};

struct AddInfluencingBones
{
    AddInfluencingBones( BonesContainer & bones, PhysiquesToBonesMap & map )
        : m_bones( bones ), m_map(map) {}

    void operator()( INode *node )
    //Add all this physique's bones!
    {
        //node will be in map iff it has a physique modifier
        PhysiquesToBonesMap::iterator p = m_map.find(node);

        if (p == m_map.end())
        {
            return; //not there
        }
       
        //it's there, add its influence bones.
        BonesContainer *influenceBones = p->second;
        
        for (BonesContainer::iterator i = influenceBones->begin();
            i != influenceBones->end(); ++i)
        {
            m_bones.insert( *i );
        }
    }

    BonesContainer & m_bones;
    PhysiquesToBonesMap & m_map;
};

struct IsntInHierarchy
{
    IsntInHierarchy( INode *theHierarchy ) : m_theHierarchy( theHierarchy ) {}

    bool operator()( INode *node )
    {
        if (FindTopOfHierarchy(node) != m_theHierarchy)
        {
            return true;
        }

        return false;
    }

    INode *m_theHierarchy;
};

RwExpError FindPhysiqueAndBipedNodes(
    Interface *gi,
    WarningList *warningList,
    INode *sceneRoot,               //input - root of entire MAX scene.
    INode *topOfSelectedHierarchy,  //input - highest node in heirarchy user selected.

    //output: all physiques we think are worth exporting
    SkinsContainer & skinsToExport,

    //output: all bones (biped parts, dummies, etc)
    //that influence any vertex in any of the above physiques
    BonesContainer & influentialBones
    )
{
    //first work out version of physique we're dealing with & get a handle to it
    s_physiqueHandle = CreatePhysiqueHandle( gi, warningList );

    if (s_physiqueHandle == 0)
    {
        return RwExpError("Sorry, couldn't determine physique version number!");
    }

    //find all physiques in selected heirarchy
    ForEachNodeInHierarchy( topOfSelectedHierarchy,
        AddToSkinsIfPhysique( skinsToExport ) );
 
    //find all bones in selected hierarchy
    BonesContainer hierarchyBones;
    
    ForEachNodeInHierarchy( topOfSelectedHierarchy,
        AddToBonesIfBipedBone( hierarchyBones) );

    //associate all physiques in scene with their influencing bones
    PhysiquesToBonesMap physiqueToBonesMap;

    ForEachNodeInHierarchy( sceneRoot,
        MapPhysiquesToInfluenceBones( physiqueToBonesMap ) );

    //Make sure all selected physique influencing bones' hierarchies are in there to heirarchy bones
    //(so we'll find all the other physiques that are attached to them!)
    std::for_each( skinsToExport.begin(), skinsToExport.end(),
        AddInfluenceBonesHeirachiesToHeirarchyBones( physiqueToBonesMap, hierarchyBones ) );

    //find & add all physiques that are influenced by some hierarchy bone
    //(and they're not in there already)
    ForEachNodeInHierarchy( sceneRoot,
        AddToSkinsIfInfluencedByBone( skinsToExport, hierarchyBones, physiqueToBonesMap ) );

    //find all the bones that influence any physique which we want export,
    //and only those bones!
    std::for_each( skinsToExport.begin(), skinsToExport.end(),
        AddInfluencingBones( influentialBones, physiqueToBonesMap ) );

    //finally check that all influencing bones are in the same heirarchy,
    //otherwise export will go wrong!
    if (!influentialBones.empty())
    {
        BonesContainer::iterator badBone =
            std::find_if( influentialBones.begin(), influentialBones.end(),
                IsntInHierarchy(
                    FindTopOfHierarchy( *(influentialBones.begin()) )
                                )
                        );

        if (badBone != influentialBones.end())
        {
            return RwExpError("Sorry, can't export this character because not all its bones are in the same hierarchy");
        }
    }

    //okay.
    return RwExpError("");
}
